//-----------------------------------------------------------------------------
//  Copyright (C) 2002-2025 Thomas S. Ullrich
//
//  This file is part of "xyscan".
//
//  This file may be used under the terms of the GNU General Public License.
//  This project is free software; you can redistribute it and/or modify it
//  under the terms of the GNU General Public License.
//  
//  Author: Thomas S. Ullrich
//  Last update: Oct 30, 2025
//-----------------------------------------------------------------------------
#include <QObject>
#include <QApplication>
#include <QCommandLineParser>
#include <QTextStream>
#include "xyscanVersion.h"
#include "xyscanWindow.h"
#include <iostream>

using namespace std;

//#define TEST_SPANISH_VERSION 1
//#define TEST_FRENCH_VERSION 1
//#define TEST_GERMAN_VERSION 1

int main(int argc, char *argv[])
{
    //
    //  Init application
    //
#if defined(Q_OS_LINUX)
    qputenv("QT_QPA_PLATFORM", "xcb");      // otherwise on Linux (Wayland) cannot use mouse grabbing the docking windows
#endif
    QApplication app(argc, argv);
    Q_INIT_RESOURCE(xyscan);
    app.setApplicationName("xyscan");
    app.setApplicationVersion(xyscanVersionString());

    //
    //  Internationalization
    //  Load translator unless we are using English.
    //  If no translator exist for the language of the host
    //  system we fall back to English.
    //
#ifdef TEST_FRENCH_VERSION
    QLocale::setDefault(QLocale(QLocale::French, QLocale::France));
#elif TEST_GERMAN_VERSION
    QLocale::setDefault(QLocale(QLocale::German, QLocale::Germany));
#elif TEST_SPANISH_VERSION
    QLocale::setDefault(QLocale(QLocale::Spanish, QLocale::Spain));
#endif
    QLocale locale;

    QTranslator qtTranslator, qtBaseTranslator, xyscanTranslator;

    if (locale.language() != QLocale::English) {
        if (qtTranslator.load(locale, "qt", "_", ":/translations")) app.installTranslator(&qtTranslator);
        if (qtBaseTranslator.load(locale, "qtbase", "_", ":/translations")) app.installTranslator(&qtBaseTranslator);
        if (xyscanTranslator.load(locale, "xyscan", "_", ":/translations")) app.installTranslator(&xyscanTranslator);
    }

    //
    //   Handle command line
    //
    QCommandLineParser parser;
    parser.setApplicationDescription(QObject::tr("xyscan, a data thief for scientific applications"));
    parser.setSingleDashWordOptionMode(QCommandLineParser::ParseAsLongOptions);
    parser.addHelpOption();
    parser.addVersionOption(); // prints e.g. "xyscan 4.6.8"
    
    QCommandLineOption numericVersionOpt(QStringList() << "numeric-version",
                                         QObject::tr("Print numeric version number and exit."));
    parser.addOption(numericVersionOpt);
    
    parser.addPositionalArgument("image", QObject::tr("Image file to scan"));
    parser.process(app);
    
    if (parser.isSet(numericVersionOpt)) {
        QTextStream(stdout) << QCoreApplication::applicationVersion() << Qt::endl;
        return 0;  // clean exit, no window shown
    }
    
    const QString imageArg = parser.positionalArguments().value(0);
    if (!imageArg.isEmpty()) {
        if (!QFile::exists(imageArg)) {
            cout << QObject::tr("File '%1' does not exist.").arg(imageArg).toStdString() << endl;
            return 2;
        }
    }

    //
    //  Create and launch xyscan
    //
    xyscanWindow win(imageArg);
    win.show();
    return app.exec();
}


